from fastapi import FastAPI, HTTPException
from pydantic import BaseModel
import httpx, os

SALAS_URL  = os.getenv("SALAS_URL",  "http://127.0.0.1:8001")
TURMAS_URL = os.getenv("TURMAS_URL", "http://127.0.0.1:8002")
NOTAS_URL  = os.getenv("NOTAS_URL",  "http://127.0.0.1:8003")

app = FastAPI(title="academico")

class AulaReq(BaseModel):
    turma_id: str
    sala: str
    inicio: str
    fim: str

class NotaReq(BaseModel):
    turma_id: str
    aluno_id: str
    avaliacao: str
    nota: float

@app.get("/health")
def health():
    return {"status": "ok"}

@app.post("/agendar-aula")
def agendar_aula(a: AulaReq):
    # 1) reserva sala
    with httpx.Client(timeout=2.0) as c:
        r = c.post(f"{SALAS_URL}/reservas", json={"sala": a.sala, "inicio": a.inicio, "fim": a.fim})
        if r.status_code not in (200, 201):
            raise HTTPException(status_code=409, detail=f"salas_error:{r.text}")
        # 2) registra aula na turma
        r2 = c.post(f"{TURMAS_URL}/aulas", json=a.dict())
        if r2.status_code != 200:
            raise HTTPException(status_code=400, detail=f"turmas_error:{r2.text}")
    return {"ok": True, "msg": "Aula agendada e sala reservada"}

@app.post("/lancar-nota")
def lancar_nota(n: NotaReq):
    # valida matrícula do aluno
    with httpx.Client(timeout=2.0) as c:
        r = c.get(f"{TURMAS_URL}/turmas/{n.turma_id}/alunos")
        if r.status_code != 200 or n.aluno_id not in r.json().get("alunos", []):
            raise HTTPException(status_code=400, detail="aluno_nao_matriculado")
        r2 = c.post(f"{NOTAS_URL}/nota", json=n.dict())
        if r2.status_code != 200:
            raise HTTPException(status_code=400, detail=f"notas_error:{r2.text}")
    return {"ok": True}
